#include <WiFi.h>
#include <PubSubClient.h>
#include <HardwareSerial.h>
#include "Adafruit_Thermal.h"

// ----------------- PINS -----------------
#define LED_PIN D1   // same as before; just for blink feedback

// ----------------- WIFI / AIO -----------------
const char* WIFI_SSID = "eiphone13";
const char* WIFI_PASS = "donkdonk";

const char* AIO_USERNAME = "ewolf"; // e.g. "ewolf"
const char* AIO_KEY      = "aio_ymtY00l42GWgmarimb2JYEJDQxp6";      // aio_xxx...
const char* MQTT_HOST    = "io.adafruit.com";
const uint16_t MQTT_PORT = 1883;

// Feed/topic
String FEED      = "oled-text";  // you can rename if you want
String TOPIC_RAW = String(AIO_USERNAME) + "/feeds/" + FEED;
String TOPIC_JSON = TOPIC_RAW + "/json";

// ----------------- OBJECTS -----------------
WiFiClient wifi;
PubSubClient mqtt(wifi);

// Use UART1 for printer, on D7 (RX), D6 (TX)
HardwareSerial mySerial(1);
Adafruit_Thermal printer(&mySerial);

// ----------------- HELPERS -----------------

void blinkLED(int n=2, int on=120, int off=90) {
  pinMode(LED_PIN, OUTPUT);
  for (int i = 0; i < n; i++) {
    digitalWrite(LED_PIN, HIGH);
    delay(on);
    digitalWrite(LED_PIN, LOW);
    delay(off);
  }
}

// Optional: simple wrapper to print a message with bottom padding
void printMessage(const String& msg) {
  printer.println(msg);
  printer.feed(3);   // extra lines so message clears the housing
}

// MQTT callback: called when a new message arrives
void onMsg(char* topic, byte* payload, unsigned int len) {
  // Copy payload into a String
  String m;
  m.reserve(len);
  for (unsigned i = 0; i < len; i++) {
    m += (char)payload[i];
  }

  // If JSON feed, extract "value" from {"value":"..."}
  String t(topic);
  if (t.endsWith("/json")) {
    int i = m.indexOf("\"value\"");
    if (i >= 0) {
      int c  = m.indexOf(':', i);
      int q1 = m.indexOf('"', c + 1);
      int q2 = m.indexOf('"', q1 + 1);
      if (q1 >= 0 && q2 > q1) {
        m = m.substring(q1 + 1, q2);
      }
    }
  }

  Serial.printf("MQTT IN [%s] (%u bytes): %s\n", topic, len, m.c_str());
  blinkLED(2);

  // Send the message to the printer
  printMessage(m);
}

// ----------------- WIFI / MQTT -----------------

void connectWiFi() {
  Serial.printf("Connecting Wi-Fi to \"%s\"...\n", WIFI_SSID);
  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - t0 < 20000) {
    delay(250);
    Serial.print(".");
  }
  Serial.println();
  if (WiFi.status() == WL_CONNECTED) {
    Serial.print("Wi-Fi OK, IP: ");
    Serial.println(WiFi.localIP());
  } else {
    Serial.println("Wi-Fi FAILED");
  }
}

bool connectMQTT() {
  mqtt.setServer(MQTT_HOST, MQTT_PORT);
  mqtt.setCallback(onMsg);
  String clientId = "xiao-c3-" + String((uint32_t)ESP.getEfuseMac(), HEX);

  Serial.printf("Connecting MQTT to %s:%u as %s\n",
                MQTT_HOST, MQTT_PORT, clientId.c_str());

  bool ok = mqtt.connect(clientId.c_str(), AIO_USERNAME, AIO_KEY);
  if (!ok) {
    Serial.printf("MQTT connect FAILED, state=%d\n", mqtt.state());
    // -2=DNS fail, -1=timeout, 4=bad user/pass, 5=not auth, 7=unavailable
    return false;
  }

  Serial.println("MQTT connected");
  Serial.println("Subscribing to:");
  Serial.println("  " + TOPIC_RAW);
  Serial.println("  " + TOPIC_JSON);
  mqtt.subscribe(TOPIC_RAW.c_str());
  mqtt.subscribe(TOPIC_JSON.c_str());
  return true;
}

unsigned long lastTry = 0;

// ----------------- SETUP / LOOP -----------------

void setup() {
  Serial.begin(115200);
  delay(200);

  // Start printer UART on D7 (RX), D6 (TX) at 19200 baud
  // RX, TX are the board macros for D7/D6
  mySerial.begin(19200, SERIAL_8N1, RX, TX);
  printer.begin();

  printer.println("Booting...");
  printer.feed(2);

  connectWiFi();
  connectMQTT();

  Serial.println("Ready. Publish to your Adafruit IO feed to print.");
}

void loop() {
  // Keep WiFi alive
  if (WiFi.status() != WL_CONNECTED && millis() - lastTry > 3000) {
    lastTry = millis();
    connectWiFi();
  }

  // Keep MQTT alive
  if (!mqtt.connected() && millis() - lastTry > 3000) {
    lastTry = millis();
    connectMQTT();
  }

  mqtt.loop();
}
